const axios = require('axios');
const cheerio = require('cheerio');

/**
 * Scrape holder data from DexScreener with more aggressive heuristics
 */
async function scrapeDexScreenerHolders(address, network) {
  try {
  // 1. Get pair address from the API first
    const apiUrl = `https://api.dexscreener.com/latest/dex/tokens/${address}`;
  console.log(`Fetching from API: ${apiUrl}`);
    
    const { data } = await axios.get(apiUrl);
    if (!data || !data.pairs || data.pairs.length === 0) {
      console.log(`No pairs found for token ${address}`);
      return { count: 0 };
    }
    
  // 2. Filter pairs matching the network and sort by liquidity
    const normalizedNetwork = network.toLowerCase() === 'eth' ? 'ethereum' : network.toLowerCase();
    let pairs = data.pairs.filter(p => p.chainId.toLowerCase() === normalizedNetwork);
    if (pairs.length === 0) pairs = data.pairs;
    
    pairs.sort((a, b) => (parseFloat(b.liquidity?.usd) || 0) - (parseFloat(a.liquidity?.usd) || 0));
    const topPair = pairs[0];
    
  // 3. Build direct pair URL
    const pairUrl = `https://dexscreener.com/${topPair.chainId}/${topPair.pairAddress}`;
    console.log(`Scraping URL: ${pairUrl}`);
    
  // 4. Fetch pair page data
    const response = await axios.get(pairUrl, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0.0.0'
      }
    });
    
  // 5. Search for holder data using several heuristics
    const $ = cheerio.load(response.data);
    
  // Log page structure to help locate holder data
  console.log("🔍 Searching for 'Holders' table on the page...");
    
    // List all tables on the page
    console.log("📊 Tables on page:");
    $('table').each((i, table) => {
      const headers = $(table).find('th').map((i, th) => $(th).text().trim()).get();
      console.log(`Table ${i}: ${headers.join(' | ')}`);
      
      // If a header contains "Holders", this is likely the table we want
      if (headers.some(h => h.toLowerCase().includes('holder'))) {
        console.log(`✅ Possible holders table found at index ${i}`);
        
        // Search table rows for possible holder numbers
        $(table).find('tbody tr').each((j, row) => {
          const cells = $(row).find('td').map((k, td) => $(td).text().trim()).get();
          console.log(`Row ${j}: ${cells.join(' | ')}`);
          
          // Look for a cell with a number that is likely the holder count
          cells.forEach(cell => {
            if (/^\d{1,3}(,\d{3})*$/.test(cell)) {
              console.log(`💡 Possible holder count found: ${cell}`);
              holderCount = parseInt(cell.replace(/,/g, ''));
            }
          });
        });
      }
    });
    
  // Log all buttons for diagnostics
    $('button').each((i, el) => {
      const className = $(el).attr('class') || '';
      const text = $(el).text().trim();
      if (text.match(/\d{3,}/) || text.toLowerCase().includes('hold')) {
        console.log(`Button ${i}: ${className} - Text: "${text}"`);
      }
    });
    
  // METHOD 1: Find a button that contains "holders" and a number
    let holderCount = 0;
    $('button').each((i, el) => {
      const text = $(el).text().trim();
      if (text.match(/\d+,?\d*\s*holders?/i) || text.match(/holders?\s*\d+,?\d*/i)) {
        const numMatch = text.match(/(\d{1,3}(,\d{3})*)/);
        if (numMatch) {
          holderCount = parseInt(numMatch[0].replace(/,/g, ''));
          console.log(`✓ FOUND: ${holderCount} from button`);
          return false;
        }
      }
    });
    
  // METHOD 2: Find divs that contain "holders" and a number
    if (!holderCount) {
      $('div').each((i, el) => {
        const text = $(el).text().trim();
        if (text.match(/\d+,?\d*\s*holders?/i) || text.match(/holders?\s*\d+,?\d*/i)) {
          const numMatch = text.match(/(\d{1,3}(,\d{3})*)/);
          if (numMatch) {
            holderCount = parseInt(numMatch[0].replace(/,/g, ''));
            console.log(`✓ FOUND: ${holderCount} from div`);
            return false;
          }
        }
      });
    }
    
  // METHOD 3: Search text inside span elements
    if (!holderCount) {
      $('span').each((i, el) => {
        const text = $(el).text().trim();
        if (text.match(/\d+,?\d*\s*holders?/i) || text.match(/holders?\s*\d+,?\d*/i)) {
          const numMatch = text.match(/(\d{1,3}(,\d{3})*)/);
            if (numMatch) {
            holderCount = parseInt(numMatch[0].replace(/,/g, ''));
            console.log(`✓ FOUND: ${holderCount} from span`);
            return false;
          }
        }
      });
    }
    
  // METHOD 4: Search anywhere on the page
    if (!holderCount) {
      const bodyText = $('body').text();
      const holderMatches = bodyText.match(/(\d{1,3}(,\d{3})*)\s*holders/i);
      if (holderMatches) {
        holderCount = parseInt(holderMatches[1].replace(/,/g, ''));
        console.log(`✓ FOUND: ${holderCount} from page text`);
      }
    }
    
    // CARA 5: Cek API transactions
    if (!holderCount && topPair.txns && topPair.txns.h24 && topPair.txns.h24.holders) {
      holderCount = topPair.txns.h24.holders;
      console.log(`✓ FOUND: ${holderCount} from API txns`);
    }
    
    return {
      count: holderCount,
      source: 'DexScreener',
      url: pairUrl
    };
  } catch (error) {
    console.error(`Error scraping: ${error.message}`);
    return { count: 0 };
  }
}

module.exports = {
  scrapeDexScreenerHolders
};
