const axios = require('axios');

/**
 * Client for interacting with the SolSniffer API
 */
async function getSolSnifferData(address) {
  try {
    console.log(`Fetching SolSniffer data for: ${address}`);
    const apiUrl = `https://solsniffer.com/api/v2/token/refresh/${address}`;
    
    const response = await axios.get(apiUrl, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0.0.0',
        'Accept': 'application/json'
      },
      timeout: 15000 // 15 second timeout
    });
    
    if (response.data && response.data.tokenData) {
      console.log('Successfully fetched SolSniffer data');
      return response.data;
    } else {
      console.log('Invalid or empty response from SolSniffer API');
      return null;
    }
  } catch (error) {
    console.error(`Error fetching SolSniffer data: ${error.message}`);
    return null;
  }
}

/**
 * Analyze token safety based on SolSniffer data
 */
function analyzeSolSnifferSecurity(solSnifferData) {
  if (!solSnifferData || !solSnifferData.tokenData) {
    return {
      isSecure: false,
      securityStatus: 'UNKNOWN',
      warnings: ['Unable to analyze token security with SolSniffer'],
      details: {}
    };
  }
  
  // Extract indicator data
  const { indicatorData, auditRisk, score } = solSnifferData.tokenData;
  
  // Parse JSON strings to objects if they're strings
  const highDetails = typeof indicatorData.high.details === 'string' ? 
    JSON.parse(indicatorData.high.details) : indicatorData.high.details;
  
  const moderateDetails = typeof indicatorData.moderate.details === 'string' ? 
    JSON.parse(indicatorData.moderate.details) : indicatorData.moderate.details;
  
  // Collect warnings based on high risk factors
  const warnings = [];
  for (const [key, value] of Object.entries(highDetails)) {
    if (value === true) {
      warnings.push(`⚠️ ${key}`);
    }
  }
  
  // Determine security status based on SolSniffer score and high risk count
  let securityStatus;
  if (score >= 80) {
    securityStatus = 'SAFE';
  } else if (score >= 65) {
    securityStatus = 'LIKELY SAFE';
  } else if (score >= 50) {
    securityStatus = 'CAUTION';
  } else {
    securityStatus = 'HIGH RISK';
  }
  
  // Check for critical security factors
  if (auditRisk) {
    if (auditRisk.mintDisabled === false) {
      warnings.unshift('⚠️ Token can be minted (supply not fixed)');
    }
    
    if (auditRisk.freezeDisabled === false) {
      warnings.unshift('⚠️ Accounts can be frozen by the authority');
    }
  }
  
  return {
    isSecure: securityStatus === 'SAFE' || securityStatus === 'LIKELY SAFE',
    securityStatus,
    warnings: warnings.slice(0, 5), // Limit to top 5 warnings
    score: score || 0,
    details: {
      highRiskCount: indicatorData.high.count,
      moderateRiskCount: indicatorData.moderate.count,
      lowRiskCount: indicatorData.low.count,
      mintable: auditRisk?.mintDisabled === false,
      freezable: auditRisk?.freezeDisabled === false,
      lpBurned: auditRisk?.lpBurned === true
    }
  };
}

module.exports = {
  getSolSnifferData,
  analyzeSolSnifferSecurity
};
