/**
 * Utilities for parsing and formatting numeric values like market cap
 */
function parseMarketCap(raw) {
  if (raw === undefined || raw === null || raw === '') return null;
  try {
    if (typeof raw === 'number') return raw;
    if (typeof raw === 'string') {
      let s = raw.replace(/\$/g, '').trim();
      s = s.replace(/,/g, '');
      const m = s.match(/^([0-9]*\.?[0-9]+)\s*([KMBkmb])?$/);
      if (m) {
        let num = parseFloat(m[1]);
        const suffix = (m[2] || '').toUpperCase();
        if (suffix === 'K') num *= 1e3;
        if (suffix === 'M') num *= 1e6;
        if (suffix === 'B') num *= 1e9;
        return num;
      }
      const parsed = parseFloat(s);
      return isNaN(parsed) ? null : parsed;
    }
    return null;
  } catch (err) {
    console.error('parseMarketCap error:', err.message);
    return null;
  }
}

// Format market cap number (or string) to readable $ with K/M/B suffix
function formatMcapReadable(raw) {
  if (raw === undefined || raw === null || raw === '') return 'Unknown';
  // If it's a string, try to parse numeric value
  let num = raw;
  if (typeof num === 'string') {
    const parsed = parseMarketCap(num);
    if (parsed !== null && !isNaN(parsed)) num = parsed;
  }
  if (typeof num !== 'number' || isNaN(num)) return String(raw);
  if (num >= 1e9) return `$${(num / 1e9).toFixed(1)}B`;
  if (num >= 1e6) return `$${(num / 1e6).toFixed(1)}M`;
  if (num >= 1e3) return `$${(num / 1e3).toFixed(1)}K`;
  return `$${Math.round(num).toLocaleString()}`;
}

module.exports = {
  parseMarketCap
  , formatMcapReadable
};
