const fs = require('fs');
const path = require('path');

/**
 * Advertisement manager for the RadarAnalyzer bot
 * This utility handles ad display and impression tracking only
 */

// Default configuration
let adConfig = {
  enabled: true,
  currentAd: 'default',
  ads: {
    'default': {
      imagePath: path.join(__dirname, '..', 'assets', 'ads.png'),
      caption: "🔸 <b>SPONSORED</b> - Want to advertise here? Contact @deepdex_support",
      buttonText: "🌟 Learn More",
      buttonUrl: "https://t.me/deepdex_official",
      impressions: 0,
      clicks: 0
    }
  }
};

// Config file path
const configFilePath = path.join(__dirname, '..', 'data', 'ad-config.json');

/**
 * Initialize the ad manager by loading configuration
 */
function initAdManager() {
  try {
    // Create data directory if it doesn't exist
    const dataDir = path.join(__dirname, '..', 'data');
    if (!fs.existsSync(dataDir)) {
      fs.mkdirSync(dataDir, { recursive: true });
    }
    
    // Load configuration file if it exists
    if (fs.existsSync(configFilePath)) {
      const configData = fs.readFileSync(configFilePath, 'utf8');
      const loadedConfig = JSON.parse(configData);
      adConfig = { ...adConfig, ...loadedConfig };
      
      console.log('Ad manager initialized with configuration from file');
    } else {
      // Create default config file
      saveAdConfig();
      console.log('Ad manager initialized with default configuration');
    }
    
    return true;
  } catch (error) {
    console.error('Error initializing ad manager:', error);
    return false;
  }
}

/**
 * Save the current ad configuration to file
 */
function saveAdConfig() {
  try {
    fs.writeFileSync(configFilePath, JSON.stringify(adConfig, null, 2), 'utf8');
    return true;
  } catch (error) {
    console.error('Error saving ad configuration:', error);
    return false;
  }
}

/**
 * Get the current ad to display
 * @returns {Object} Ad configuration for the current ad
 */
function getCurrentAd() {
  if (!adConfig.enabled) {
    return null;
  }
  
  const adId = adConfig.currentAd;
  const ad = adConfig.ads[adId];
  
  if (!ad) {
    return adConfig.ads['default'];
  }
  
  // Check if the image exists
  if (!fs.existsSync(ad.imagePath)) {
    console.error(`Ad image not found: ${ad.imagePath}`);
    return adConfig.ads['default'];
  }
  
  return ad;
}

/**
 * Record an ad impression
 * @param {string} adId - ID of the ad that was shown
 */
function recordImpression(adId = adConfig.currentAd) {
  if (adConfig.ads[adId]) {
    adConfig.ads[adId].impressions++;
    saveAdConfig();
  }
}

/**
 * Record an ad click
 * @param {string} adId - ID of the ad that was clicked
 */
function recordClick(adId = adConfig.currentAd) {
  if (adConfig.ads[adId]) {
    adConfig.ads[adId].clicks++;
    saveAdConfig();
  }
}

// Initialize when module is loaded
initAdManager();

module.exports = {
  getCurrentAd,
  recordImpression,
  recordClick
};
