const axios = require('axios');
const address = process.argv[2];
const preferredNetwork = process.argv[3]; // optional: 'bsc' | 'ethereum' | 'solana'

if (!address) {
  console.error('Usage: node scripts/check_mcap.js <address> [preferredNetwork]');
  process.exit(1);
}

async function fetchDexScreener(addr) {
  const url = `https://api.dexscreener.com/latest/dex/tokens/${addr}`;
  console.log('Fetching DexScreener:', url);
  const res = await axios.get(url, { timeout: 15000 });
  return res.data;
}

function printPair(p) {
  console.log('--- Pair ---');
  console.log('chainId:', p.chainId);
  console.log('pairAddress:', p.pairAddress);
  console.log('mcap:', p.mcap);
  console.log('fdv:', p.fdv);
  console.log('priceUsd:', p.priceUsd);
  console.log('priceNative:', p.priceNative);
  console.log('liquidity usd:', p.liquidity?.usd);
  console.log('volume h24:', p.volume?.h24);
  console.log('txns h24 holders:', p.txns?.h24?.holders);
}

(async () => {
  try {
    const data = await fetchDexScreener(address);
    if (!data || !data.pairs || data.pairs.length === 0) {
      console.error('No pairs returned by DexScreener');
      return;
    }

    console.log(`Total pairs: ${data.pairs.length}`);

    // If preferred network provided, try to find pair for it
    let chosenPair = null;
    if (preferredNetwork) {
      const norm = preferredNetwork.toLowerCase();
      chosenPair = data.pairs.find(p => p.chainId && p.chainId.toLowerCase().includes(norm));
    }

    if (!chosenPair) {
      // Fallback: take pair with highest liquidity
      const sorted = data.pairs.slice().sort((a,b) => (parseFloat(b.liquidity?.usd)||0)-(parseFloat(a.liquidity?.usd)||0));
      chosenPair = sorted[0];
    }

    console.log('\nChosen pair summary:');
    printPair(chosenPair);

    console.log('\nAll pairs (summary):');
    data.pairs.forEach((p, idx) => {
      console.log(`#${idx+1} ${p.chainId} | mcap:${p.mcap} | fdv:${p.fdv} | priceUsd:${p.priceUsd} | liquidity:${p.liquidity?.usd}`);
    });

    // Try to compute mcap from price * supply if fdv or mcap seems off
    // Attempt to load token details from services (may fail if RPC or API keys missing)
    let tokenDetails = null;
    try {
      const { getBscTokenDetails } = require('../services/bsc');
      const { getEthTokenDetails } = require('../services/ethereum');
      const { getSolanaTokenDetails } = require('../services/solana');

      // Try each based on chainId
      if (chosenPair.chainId && chosenPair.chainId.toLowerCase().includes('bsc')) {
        tokenDetails = await getBscTokenDetails(address);
      } else if (chosenPair.chainId && (chosenPair.chainId.toLowerCase().includes('eth') || chosenPair.chainId.toLowerCase().includes('ethereum'))) {
        tokenDetails = await getEthTokenDetails(address);
      } else if (chosenPair.chainId && chosenPair.chainId.toLowerCase().includes('solana')) {
        tokenDetails = await getSolanaTokenDetails(address);
      }
    } catch (err) {
      console.warn('Could not fetch tokenDetails via services (this may require RPC keys). Error:', err.message);
    }

    if (tokenDetails) {
      console.log('\nToken details fetched:');
      console.log('totalSupply:', tokenDetails.totalSupply);
      const price = parseFloat(chosenPair.priceUsd || chosenPair.price);
      const supply = parseFloat(tokenDetails.totalSupply);
      if (!isNaN(price) && !isNaN(supply)) {
        console.log(`Computed MCAP = price * supply = ${price} * ${supply} = ${price * supply}`);
      } else {
        console.log('Could not compute MCAP because price or supply is missing/NaN');
      }
    } else {
      console.log('\nNo tokenDetails available to compute MCAP. You can provide RPC/API keys in environment and try again.');
    }

    console.log('\nRaw DexScreener response snippet:');
    console.log(JSON.stringify(data.pairs.map(p=>({chainId:p.chainId,pairAddress:p.pairAddress,mcap:p.mcap,fdv:p.fdv,priceUsd:p.priceUsd,liquidity:p.liquidity?.usd})), null, 2));

    console.log('\nNotes:');
    console.log('- DexScreener mcap (field `mcap`) is provided by DexScreener and may represent an estimated circulating market cap or a particular pair-derived calculation.');
    console.log('- Differences vs other sources can stem from: differing circulating supply assumptions, using FDV instead of MCAP, or inconsistent pair selection (different chains/pairs).');

  } catch (error) {
    console.error('Error checking MCAP:', error.message);
  }
})();
