require('dotenv').config();
const express = require('express');
const app = express();
const PORT = 3000;
const BIRDEYE_API_KEY = process.env.BIRDEYE_API_KEY;
// Function to fetch token ATH from Birdeye
async function getTokenATH(address, network) {
  // Birdeye only supports Solana. Endpoint: https://public-api.birdeye.so/public/defi/v3/token/price_stats?address=<address>
  if (network !== 'SOL') return null;
  try {
    const url = `https://public-api.birdeye.so/public/defi/v3/token/price_stats?address=${address}`;
    const res = await axios.get(url, {
      headers: { 'x-api-key': BIRDEYE_API_KEY }
    });
    if (res.data && res.data.data && res.data.data.ath_price) {
      return res.data.data.ath_price;
    }
  } catch (err) {
    console.error('[Birdeye] Error fetching ATH:', err.message);
  }
  return null;
}
const TelegramBot = require('node-telegram-bot-api');
const { analyzeToken } = require('./services/analyzer');
const { detectNetwork } = require('./utils/networkDetector');
const path = require('path');
const fs = require('fs');
const axios = require('axios');
const { createDefaultAssets } = require('./utils/createDefaultAssets');
const { getFirstCall, setFirstCall, getLatestScan } = require('./utils/firstCallDb');
const { generatePnlCard } = require('./utils/imageGenerator');
const { parseMarketCap, formatMcapReadable } = require('./utils/numberUtils');
const { getTopHolders, buildTelegramMessageAsync } = require('./utils/solscanHolders');

// Create default assets before starting the bot
createDefaultAssets();

// Initialize the bot
const bot = new TelegramBot(process.env.TELEGRAM_BOT_TOKEN, { polling: true });
console.log('RadarAnalyzer bot is running...');
// Resolve bot username so we can build an "Add to Group" link
let botUsername = process.env.BOT_USERNAME || null;
bot.getMe().then(info => {
  if (info && info.username) {
    botUsername = info.username;
    console.log('Bot username:', botUsername);
  }
}).catch(err => {
  console.warn('Could not fetch bot info (getMe):', err && err.message);
});

// Helper: always return the username of the message sender (msg.from), never the chat/group or bot username
function getCallerUsername(msg) {
  if (!msg || !msg.from) return 'Unknown';
  if (msg.from.username) return `@${msg.from.username}`;
  // Fallback to first_name + last_name if username not available
  const parts = [];
  if (msg.from.first_name) parts.push(msg.from.first_name);
  if (msg.from.last_name) parts.push(msg.from.last_name);
  if (parts.length) return parts.join(' ');
  return 'Unknown';
}

// Simple per-chat processing lock to avoid stacked replies when multiple messages arrive quickly
const activeChats = new Map();
function acquireLock(chatId) {
  const key = String(chatId);
  if (activeChats.get(key)) return false;
  activeChats.set(key, true);
  return true;
}
function releaseLock(chatId) {
  activeChats.delete(String(chatId));
}

// Welcome message
bot.onText(/\/start/, (msg) => {
  const chatId = msg.chat.id;
  // Username of the user who invoked /start (always from msg.from)
  const callerUsername = getCallerUsername(msg);
  const welcomeText = 
    `Welcome to RadarAnalyzer Bot! 🚀\n\n` +
    `Hello ${callerUsername}!\n\n` +
    `I can help you analyze tokens across multiple blockchains:\n` +
    `- Binance Smart Chain (BSC)\n` +
    `- Ethereum (ETH)\n` +
    `- Solana (SOL)\n\n` +
    `To analyze a token, simply send me the contract address.`;
  
  // Create inline keyboard with proper formatting
  // Provide an "Add to Group" button that opens the Telegram UI to add this bot to a group
  // Sanitize bot username (remove leading @ if any) and always include ?startgroup=true so Telegram opens the group chooser
  const sanitizedBotUsername = botUsername ? botUsername.replace(/^@/, '') : null;
  const addToGroupUrl = sanitizedBotUsername ? `https://t.me/${sanitizedBotUsername}?startgroup=true` : '#';
  const inlineKeyboard = {
    reply_markup: {
      inline_keyboard: [
        [
          { text: 'Add to Group', url: addToGroupUrl }
        ]
      ]
    }
  };
  console.log('Sending welcome message with Add to Group button');
  
  // Path to welcome image
  const imagePath = path.join(__dirname, 'assets', 'cover-deepdex-token-analyzer.png');
  
  // Check if image exists
  if (fs.existsSync(imagePath)) {
    // Send image with caption and inline keyboard
    bot.sendPhoto(
      chatId, 
      imagePath, 
      {
        caption: welcomeText,
        ...inlineKeyboard
      }
    ).then(() => {
      console.log('Welcome message with image and button sent successfully');
    }).catch(error => {
      console.error('Error sending welcome message with image:', error);
    });
  } else {
    // Fallback to text message with inline keyboard if image not found
    console.log('Welcome image not found at:', imagePath);
    bot.sendMessage(
      chatId, 
      welcomeText, 
      inlineKeyboard
    ).then(() => {
      console.log('Welcome message with button sent successfully');
    }).catch(error => {
      console.error('Error sending welcome message:', error);
    });
  }
});

// Help command
bot.onText(/\/help/, (msg) => {
  const chatId = msg.chat.id;
  bot.sendMessage(
    chatId,
    `RadarAnalyzer Bot Commands:\n\n` +
    `/start - Start the bot\n` +
    `/help - Display this help message\n` +
    `/analyze <address> - Analyze a specific token\n` +
    `/th <address> - Display top 10 token holders\n` +
    `Or simply send a contract address to analyze it.`
  );
});

// Analyze command
bot.onText(/\/analyze (.+)/, async (msg, match) => {
  const chatId = msg.chat.id;
  // Check if match[1] exists before calling trim()
  const address = match && match[1] ? match[1].trim() : '';
  
  // Only process if address is not empty
  if (address) {
    await handleTokenAnalysis(chatId, address, msg.message_id, '', null, null, getCallerUsername(msg));
  } else {
    bot.sendMessage(chatId, "❌ Please provide a valid contract address. Example: /analyze 0x123...");
  }
});

// /flex command: generate PnL card comparing first call Mcap to latest Mcap
bot.onText(/\/flex\s+(.+)/i, async (msg, match) => {
  const chatId = msg.chat.id;
  // avoid concurrent flex requests per chat
  if (!acquireLock(chatId)) {
  try { bot.sendChatAction(chatId, 'typing'); } catch (e) {}
  return bot.sendMessage(chatId, '⚠️ Please wait, processing previous request...');
  }
  try {
  const mint = match && match[1] ? match[1].trim().replace(/\s+/g, '') : null;
    if (!mint) {
    releaseLock(chatId);
    return bot.sendMessage(chatId, '❌ Please provide a contract address. Example: /flex 0x123...');
  }
    try { bot.sendChatAction(chatId, 'typing'); } catch (e) {}
    const callerUsername = getCallerUsername(msg);
    const network = detectNetwork(mint);
    if (!network) return bot.sendMessage(chatId, '❌ Could not detect network for that address.');

    // Determine group key (if group chat use chat id)
    let groupKey = null;
    if (msg.chat && (msg.chat.type === 'group' || msg.chat.type === 'supergroup')) {
      groupKey = `${chatId}`;
    }

    // Fetch DB rows
    const firstRow = await getFirstCall(groupKey, mint);
    if (!firstRow) {
      return bot.sendMessage(chatId, `ℹ️ This token hasn't been called in this chat yet. Please call it first by sending the contract address or using /analyze ${mint}. After a second call the PnL card will be available.` , { reply_to_message_id: msg.message_id });
    }
    const latestRow = await getLatestScan(groupKey, mint);
    if (!latestRow || (latestRow && latestRow.scanned_at && firstRow.first_called_at && latestRow.scanned_at <= firstRow.first_called_at)) {
      return bot.sendMessage(chatId, `ℹ️ Only one call found for this token in this chat (first by ${firstRow.username_called || 'unknown'}). Send/call it again later to generate PnL card.`, { reply_to_message_id: msg.message_id });
    }

    // Try to get symbol using analyzer (non-blocking if it fails)
    let symbol = '';
    try {
      const info = await analyzeToken(mint, network, null, callerUsername);
      symbol = info && info.symbol ? info.symbol : '';
    } catch (e) {
      // ignore errors from analyzer here
    }

    // Build card
    const buffer = await generatePnlCard({
      symbol,
      firstMcapRaw: firstRow.first_mcap,
      latestMcapRaw: latestRow.first_mcap,
      firstCalledBy: firstRow.username_called,
      firstCalledAt: firstRow.first_called_at,
      latestCalledAt: latestRow.scanned_at
      , marginTop: 80
    });

    if (!buffer) {
      return bot.sendMessage(chatId, '❌ Failed to generate PnL card.');
    }

    await bot.sendPhoto(chatId, buffer, { caption: `<b>PNL: ${symbol || mint}</b>`, parse_mode: 'HTML', reply_to_message_id: msg.message_id });
  } catch (err) {
    console.error('[flex] error:', err);
    bot.sendMessage(chatId, `❌ Error generating PnL card: ${err.message}`, { reply_to_message_id: msg.message_id });
  } finally {
    releaseLock(chatId);
  }
});

// /th command: Top Holders (Solana token only)
bot.onText(/\/th\s+(.+)/i, async (msg, match) => {
  const chatId = msg.chat.id;
  if (!acquireLock(chatId)) {
    try { bot.sendChatAction(chatId, 'typing'); } catch (e) {}
    return bot.sendMessage(chatId, '⚠️ Please wait, processing previous request...');
  }
  try {
    try { bot.sendChatAction(chatId, 'typing'); } catch (e) {}
    const mint = match && match[1] ? match[1].trim().replace(/\s+/g, '') : null;
    if (!mint) return bot.sendMessage(chatId, '❌ Please provide a token mint address. Example: /th <mint>');

    // Only support Solana for now (links go to solscan)
    const network = detectNetwork(mint);
    if (network !== 'SOL') {
      return bot.sendMessage(chatId, '❌ /th command currently supports Solana mints only.');
    }

    // Try to get a nicer title using token symbol (non-blocking)
    let symbol = '';
    try {
      const info = await analyzeToken(mint, 'SOL', null, getCallerUsername(msg));
      symbol = info && info.symbol ? info.symbol : '';
    } catch (e) {
      // ignore errors from analyzer
    }

    // Fetch holders (use RPC primary; cache reduces repeated load). No diagnostics sent to chat.
    const _res = await getTopHolders(mint, 10); // get a bit more and then format top 10
    // result may be an array (legacy) or an object { items, diagnostics }
    let top = Array.isArray(_res) ? _res : (_res && _res.items ? _res.items : []);
    if (!top || top.length === 0) {
      return bot.sendMessage(chatId, `ℹ️ Could not fetch top holders for ${mint}.`);
    }

    // Use the symbol (if available) for a nicer title
    const titleName = symbol && symbol.length ? symbol : (mint || '').slice(0, 8);

    // Build Telegram-ready HTML using the shared formatter (avoid double-fetch)
    const itemsToShow = top.slice(0, 10).map((it, idx) => ({
      rank: it.rank || (idx + 1),
      address: it.address || it.owner || it.key || it.pubkey,
      amount: it.amount || it.uiAmount || it.balance,
      percent: it.percent,
      raw: it.raw || it
    }));

    const telegramText = await buildTelegramMessageAsync(mint, itemsToShow, { titleEmoji: '💎', titleName, tokenSymbol: symbol });
    await bot.sendMessage(chatId, telegramText, { parse_mode: 'HTML', disable_web_page_preview: true });
  } catch (err) {
    console.error('[th] error:', err && err.message);
    bot.sendMessage(chatId, `❌ Error fetching top holders: ${err.message || err}`);
  } finally {
    releaseLock(chatId);
  }
});

// Helper: format large numbers with suffixes (K/M/B)
function formatNumberShort(num) {
  if (num === undefined || num === null) return '0';
  const n = Number(num);
  if (isNaN(n)) return String(num);
  if (n >= 1e9) return (n / 1e9).toFixed(2) + 'B';
  if (n >= 1e6) return (n / 1e6).toFixed(2) + 'M';
  if (n >= 1e3) return (n / 1e3).toFixed(2) + 'K';
  return n.toString();
}

// Handle direct address messages
bot.on('message', async (msg) => {
  // Skip commands
  if (msg.text && msg.text.startsWith('/')) return;
  const chatId = msg.chat.id;
  if (!msg.text) return;
  // Prevent overlapping processing for the same chat
    if (!acquireLock(chatId)) {
    // Notify user briefly that a request is being processed
    try { bot.sendChatAction(chatId, 'typing'); } catch (e) {}
    return bot.sendMessage(chatId, '⚠️ Please wait, processing previous request...');
  }
  try {
    // Indicate typing immediately
    try { bot.sendChatAction(chatId, 'typing'); } catch (e) {}
  // Clean up address: trim, remove whitespace/newlines
  const possibleAddress = msg.text.replace(/\s+/g, '').trim();
  console.log('Received address:', possibleAddress);
  // Simple validation to check if it might be an address
  if (possibleAddress.length >= 30) {
  // If in a group, include caller info, market cap, and ATH
  let analysis = null;
  let calledByLine = '';
  let firstCallLine = '';
  let groupId = null;
  let athValue = null;
    if (msg.chat.type === 'group' || msg.chat.type === 'supergroup') {
      groupId = `${chatId}`;
      try {
        const network = detectNetwork(possibleAddress);
        const callerUsername = getCallerUsername(msg);
        if (!network) return;
        // Ensure username of caller is available to analyzer (used when saving scanned token)
        analysis = await analyzeToken(possibleAddress, network, groupId, callerUsername);
  // Fetch token ATH from Birdeye if network is Solana
  athValue = await getTokenATH(possibleAddress, network);
  // Ensure caller username is taken from the message sender (msg.from) and not chat or bot
  const username = getCallerUsername(msg);
  // Fetch marketCap from DexScreener if available, fallback to FDV if not
        let mcap = 'Unknown';
        console.log('[GROUP DEBUG] analysis.marketCap:', analysis.marketCap, 'analysis.mcap:', analysis.mcap, 'analysis.fdv:', analysis.fdv);
        if (typeof analysis.marketCap === 'number' && !isNaN(analysis.marketCap)) {
          mcap = analysis.marketCap;
        } else if (typeof analysis.mcap === 'number' && !isNaN(analysis.mcap)) {
          mcap = analysis.mcap;
        } else if (typeof analysis.fdv === 'number' && !isNaN(analysis.fdv)) {
          mcap = analysis.fdv;
        } else {
          // Fallback: coba parse dari analysis.text jika ada baris "M-Cap: $..."
          if (analysis.text) {
            const mcapMatch = analysis.text.match(/M[- ]?Cap:\s*\$([0-9.,A-Za-z]+)/i);
            if (mcapMatch && mcapMatch[1]) {
              mcap = `$${mcapMatch[1]}`;
              console.log('[GROUP DEBUG] Fallback Mcap from text:', mcap);
            }
          }
        }
        // Normalize mcap into numeric (float) for storage and a display string for captions
        let mcapNum = null;
        if (typeof mcap === 'number' && !isNaN(mcap)) {
          mcapNum = mcap;
        } else if (typeof mcap === 'string' && mcap !== 'Unknown') {
          mcapNum = parseMarketCap(mcap);
        }
  const mcapDisplay = (mcapNum !== null && !isNaN(mcapNum)) ? formatMcapReadable(mcapNum) : (typeof mcap === 'string' ? formatMcapReadable(mcap) : 'Unknown');
        // PostgreSQL-backed first call Mcap logic
        const groupKey = groupId;
        const caKey = possibleAddress;
        let firstMcap = null;
        let groupLink = null;
  let usernameCalled = username;
  let dbRow = null;
        // Try to get group invite link (if bot is admin)
        if (msg.chat && msg.chat.type && (msg.chat.type === 'group' || msg.chat.type === 'supergroup')) {
          // Try to get group invite link via getChat if not present
          if (msg.chat.invite_link) {
            groupLink = msg.chat.invite_link;
          } else {
            try {
              const chatInfo = await bot.getChat(chatId);
              if (chatInfo && chatInfo.invite_link) {
                groupLink = chatInfo.invite_link;
              }
            } catch (e) {
              // ignore
            }
          }
        }
        try {
          dbRow = await getFirstCall(groupKey, caKey);
          console.log('[DB DEBUG] First call row:', dbRow);
        } catch (dbErr) {
          console.error('Error fetching first call Mcap from DB:', dbErr);
        }
        // Always attempt to read latestRow as a separate step so variable is defined
        let latestRow = null;
        try {
          latestRow = await getLatestScan(groupKey, caKey);
          console.log('[DB DEBUG] Latest call row:', latestRow);
        } catch (err) {
          console.error('[DB DEBUG] Error fetching latest call:', err);
        }
        if (!firstMcap && mcapNum !== null && !isNaN(mcapNum)) {
          try {
            // save numeric marketcap (float) without $ sign
            await setFirstCall(groupKey, caKey, mcapNum, groupLink, usernameCalled, network, Date.now()/1000);
            firstMcap = mcapNum;
          } catch (dbErr) {
            console.error('Error setting first call Mcap in DB:', dbErr);
          }
        }
  // Always display first call info from DB (username unchanged)
        if (dbRow && dbRow.username_called) {
          // Always display two lines: first call and latest call
          const firstCallMcapDisplay = dbRow && dbRow.first_mcap ? formatMcapReadable(dbRow.first_mcap) : 'Unknown';
          const latestCallMcapDisplay = latestRow && latestRow.first_mcap ? formatMcapReadable(latestRow.first_mcap) : 'Unknown';
          const firstCallLine = `First call by: ${dbRow.username_called} | Mcap: ${firstCallMcapDisplay}`;
          const nowCallLine = latestRow ? `Called now by: ${latestRow.username_called} | Mcap: ${latestCallMcapDisplay}` : '';
          calledByLine = `\n${firstCallLine}\n${nowCallLine}\n`;
        } else {
          // New contract address: show current scan info as first and latest
          // For a new CA, first_mcap = mcapStr
          // Build display strings for first and now Mcap
          let firstCallMcapDisplay = mcapDisplay;
          if (dbRow && dbRow.first_mcap !== undefined && dbRow.first_mcap !== null) {
            firstCallMcapDisplay = formatMcapReadable(dbRow.first_mcap);
          }
          const firstCallLine = `First call by: ${username} | Mcap: ${firstCallMcapDisplay}`;
          const nowCallLine = `Called now by: ${username} | Mcap: ${mcapDisplay}`;
          calledByLine = `\n${firstCallLine}\n${nowCallLine}\n`;
        }
      } catch (e) {
        calledByLine = `\nError fetching Mcap: ${e.message}`;
      }
      finally {
        // Clear the global username to avoid leaking between requests after processing
        // We'll clear it here AFTER DB/setFirstCall in this try block completes (setFirstCall is called below)
      }
    } else {
    // Private chat: still save to DB without groupId
      let network = detectNetwork(possibleAddress);
      let athValuePrivate = null;
      try {
        if (!network) return;
        const callerUsernamePrivate = getCallerUsername(msg);
        // Make caller username available to analyzer/save routines
        analysis = await analyzeToken(possibleAddress, network, null, callerUsernamePrivate);
        athValuePrivate = await getTokenATH(possibleAddress, network);
      } catch (e) {
        // fallback: biarkan error ditangani di handleTokenAnalysis
      }
  await handleTokenAnalysis(chatId, possibleAddress, msg.message_id, calledByLine, analysis, athValuePrivate, getCallerUsername(msg));
      return;
    }
  await handleTokenAnalysis(chatId, possibleAddress, msg.message_id, calledByLine, analysis, athValue, getCallerUsername(msg));
    // ATH will be merged into the main message and not sent separately
  } else {
    // Do not reply to non-address messages. Per rule: only respond to commands or valid addresses.
    return;
  }
  } finally {
    releaseLock(chatId);
  }
});

async function handleTokenAnalysis(chatId, address, message_id, calledByLine = '', preAnalysis = null, athValue = null, usernameCalled = null) {
  try {
    // Send a "typing..." action
    bot.sendChatAction(chatId, 'typing');
    // Detect which network the address belongs to
    const network = detectNetwork(address);
    if (!network) {
      console.error('[handleTokenAnalysis] Address not detected as valid network:', address);
      bot.sendMessage(chatId, `❌ Invalid address format. Please provide a valid contract address.\n\nDebug: ${address}`, {
        reply_to_message_id: message_id
      });
      return;
    }
    // Inform user that analysis has started with correct network
    const loadingMsg = await bot.sendMessage(chatId, `🔍 Analyzing token on ${network}: ${address}\nPlease wait...`, {
      reply_to_message_id: message_id
    });

  // Perform the analysis. Pass through caller username if provided to ensure saves use correct user.
  const callerForAnalysis = usernameCalled || null;
  const analysis = preAnalysis || await analyzeToken(address, network, null, callerForAnalysis);
    const refinedNetwork = analysis.network || network;

    // Determine which security image to send based on the analysis status
    const securityStatus = analysis.securityStatus || 'UNKNOWN';
    // Map security status to image file name
    let securityImageFile;
    switch(securityStatus) {
      case 'SAFE':
        securityImageFile = 'img-safe.png';
        break;
      case 'LIKELY SAFE':
        securityImageFile = 'img-likely-safe.png';
        break;
      case 'CAUTION':
        securityImageFile = 'img-caution.jpg';
        break;
      case 'HIGH RISK':
        securityImageFile = 'img-high-risk.jpg';
        break;
      default:
        securityImageFile = 'img-unknown.png';
    }
    // Create full path to the image
    const imagePath = path.join(__dirname, 'assets', securityImageFile);
    // Create inline keyboard with trading bot links and network-specific information
    const maestroLink = `https://t.me/maestro?start=${analysis.contractAddress}-abxprt`;
    const sigmaLink = `https://t.me/Sigma_buyBot?start=x780283668-${analysis.contractAddress}`;
    // Add network-specific block explorer link
    let explorerUrl = '#';
    switch(refinedNetwork) {
      case 'BSC':
        explorerUrl = `https://bscscan.com/token/${address}`;
        break;
      case 'ETH':
        explorerUrl = `https://etherscan.io/token/${address}`;
        break;
      case 'SOL':
        explorerUrl = `https://solscan.io/token/${address}`;
        break;
    }
    const inlineKeyboard = {
      reply_markup: {
        inline_keyboard: []
      }
    };

  // Merge calledByLine into the analysis message (just below the social section)
    let analysisText = analysis.text || '';
    let athText = '';
    if (athValue) {
      athText = `🏆 ATH Token: $${athValue.toLocaleString()}`;
    }
    if (calledByLine || athText) {
  // Append ATH below calledByLine so it always appears in the main output
      let insertText = calledByLine;
      if (athText) insertText += `\n${athText}\n`;
      // Insert after the last line containing 'Social' or append if not present
      const socialIdx = analysisText.toLowerCase().lastIndexOf('social');
      if (socialIdx !== -1) {
        // Find the end of the 'Social...' line
        const nextNewline = analysisText.indexOf('\n', socialIdx);
        if (nextNewline !== -1) {
          analysisText = analysisText.slice(0, nextNewline + 1) + insertText + analysisText.slice(nextNewline + 1);
        } else {
          analysisText += insertText;
        }
      } else {
        analysisText += insertText;
      }
    }

    if (fs.existsSync(imagePath)) {
      // Fix for caption too long error - split message into parts
      const maxCaptionLength = 1024; // Telegram's limit is around 1024 characters
      let shortCaption = analysisText;
      if (analysisText.length > maxCaptionLength) {
        shortCaption = analysisText.substring(0, maxCaptionLength - 150) + "\n\n... (continued in next message) ...";
      }
      await bot.deleteMessage(chatId, loadingMsg.message_id).catch(err => {
        console.log("Could not delete loading message: ", err.message);
      });
      await bot.sendPhoto(
        chatId,
        imagePath,
        {
          caption: shortCaption,
          parse_mode: 'HTML',
          disable_web_page_preview: true,
          reply_to_message_id: message_id,
          ...inlineKeyboard
        }
      ).catch(error => {
        console.error('Error sending analysis with image:', error);
        bot.sendMessage(
          chatId,
          analysisText.substring(0, 4000),
          {
            parse_mode: 'HTML',
            disable_web_page_preview: true,
            reply_to_message_id: message_id,
            ...inlineKeyboard
          }
        );
      });
      if (analysisText.length > maxCaptionLength) {
        const remainingText = analysisText.substring(maxCaptionLength - 150);
        if (remainingText.trim().length > 0) {
          await bot.sendMessage(
            chatId,
            "📊 <b>Additional Details:</b>\n\n" + remainingText,
            {
              parse_mode: 'HTML',
              disable_web_page_preview: true
            }
          ).catch(err => console.error('Error sending additional details:', err));
        }
      }
    } else {
      await bot.deleteMessage(chatId, loadingMsg.message_id).catch(err => {
        console.log("Could not delete loading message: ", err.message);
      });
      await bot.sendMessage(
        chatId,
        analysisText,
        {
          parse_mode: 'HTML',
          disable_web_page_preview: true,
          reply_to_message_id: message_id,
          ...inlineKeyboard
        }
      );
    }
  } catch (error) {
    console.error('Error analyzing token:', error);
    bot.sendMessage(
      chatId, 
      `❌ Error analyzing token: ${error.message}`, 
      { reply_to_message_id: message_id }
    );
  }
}

// Error handling
process.on('unhandledRejection', (reason, promise) => {
  console.error('Unhandled Rejection at:', promise, 'reason:', reason);
});

process.on('uncaughtException', (error) => {
  console.error('Uncaught Exception:', error);
});

app.use(express.json());
// Middleware untuk menangani error 503 (Service Unavailable)
  app.use("/maintenance", (req, res) => {
    res.status(503).json({
      error: "Service Unavailable",
      message: "Server is under maintenance, please try again later.",
    });
  });
  
  // Middleware to handle 404 (Not Found)
  app.use((req, res) => {
    res.status(404).json({
      error: "Not Found",
      message: "The page or endpoint you are looking for was not found.",
    });
  });
  app.get('/', (req, res) => {
    // console.log(`Rendering a cool ascii face for route '/cool'`)
    res.send(cool())
  })

  process.on('SIGTERM', async () => {
    // console.log('SIGTERM signal received: gracefully shutting down');
    // Perform any necessary cleanup here
    // For example, close database connections, stop background tasks, etc.
    if (server) {
        server.close(() => {
            // console.log('HTTP server closed');
            // Exit the process after the server is closed
            process.exit(0);
        });
    } else {
        // If there is no server, exit the process immediately
        process.exit(0);
    }
});
  
  // Start server
  app.listen(PORT, () => {
    // console.log(`Server running at http://localhost:${PORT}`);
  });